import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:viserpay/core/helper/date_converter.dart';
import 'package:viserpay/core/route/route.dart';
import 'package:viserpay/core/utils/dimensions.dart';
import 'package:viserpay/core/utils/my_color.dart';
import 'package:viserpay/core/utils/my_strings.dart';
import 'package:viserpay/core/utils/style.dart';
import 'package:viserpay/core/utils/url_container.dart';
import 'package:viserpay/core/utils/util.dart';
import 'package:viserpay/data/controller/nominee/nominee_controller.dart';
import 'package:viserpay/data/controller/savings/savings_controller.dart';
import 'package:viserpay/data/repo/nominee/nominee_repo.dart';
import 'package:viserpay/view/components/app-bar/custom_appbar.dart';
import 'package:viserpay/view/components/bottom-sheet/custom_bottom_sheet.dart';
import 'package:viserpay/view/components/buttons/gradient_rounded_button.dart';
import 'package:viserpay/view/components/image/my_image_widget.dart';
import 'package:viserpay/view/components/no_data.dart';
import 'package:viserpay/view/components/shimmer/faq_shimmer.dart';
import 'package:viserpay/view/components/snack_bar/show_custom_snackbar.dart';
import 'package:viserpay/view/screens/nominee/widget/nominee_card.dart';
import 'package:viserpay/view/screens/nominee/widget/nominee_remove_bottom_sheet.dart';

class NomineeListScreen extends StatefulWidget {
  const NomineeListScreen({super.key});

  @override
  State<NomineeListScreen> createState() => _NomineeListScreenState();
}

class _NomineeListScreenState extends State<NomineeListScreen> {
  @override
  void initState() {
    Get.put(NomineeRepo(apiClient: Get.find()));
    Get.put(NomineeController(nomineeRepo: Get.find()));
    super.initState();
    WidgetsBinding.instance.addPostFrameCallback((timeStamp) {
      Get.find<NomineeController>().getNomineeList();
    });
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: CustomAppBar(title: MyStrings.savings, isTitleCenter: true),
      body: GetBuilder<NomineeController>(
        builder: (controller) {
          return Padding(
            padding: Dimensions.defaultPaddingHV,
            child: Column(
              children: [
                InkWell(
                  onTap: () {
                    Get.toNamed(RouteHelper.addNewNomineeScreen);
                  },
                  child: Container(
                    padding: EdgeInsets.symmetric(horizontal: Dimensions.space15, vertical: Dimensions.space15),
                    width: double.infinity,
                    decoration: BoxDecoration(
                      color: MyColor.primaryColor.withValues(alpha: 0.1),
                      borderRadius: BorderRadius.circular(Dimensions.largeRadius),
                      border: Border.all(color: MyColor.primaryColor, width: .5),
                    ),
                    child: Row(
                      children: [
                        Container(
                          padding: EdgeInsets.all(Dimensions.space10),
                          decoration: BoxDecoration(
                            color: MyColor.primaryColor.withValues(alpha: 0.1),
                            shape: BoxShape.circle,
                            border: Border.all(color: MyColor.primaryColor, width: .5),
                          ),
                          child: Icon(Icons.person_add, color: MyColor.primaryColor),
                        ),
                        const SizedBox(width: Dimensions.space20),
                        Expanded(
                          child: Column(
                            crossAxisAlignment: CrossAxisAlignment.start,
                            children: [
                              Text(MyStrings.addNewNominee, style: semiBoldDefault.copyWith(color: MyColor.primaryColor), maxLines: 1, overflow: TextOverflow.ellipsis),
                              Text(MyStrings.nomineeRequired.tr, style: lightDefault.copyWith(), maxLines: 3, overflow: TextOverflow.ellipsis),
                            ],
                          ),
                        ),
                      ],
                    ),
                  ),
                ),
                const SizedBox(height: Dimensions.space20),
                if (controller.isLoading) ...[
                  Expanded(
                    child: ListView.separated(
                      separatorBuilder: (context, index) => SizedBox(height: 10),
                      itemCount: 10,
                      itemBuilder: (BuildContext context, int index) {
                        return Container(
                          padding: EdgeInsets.symmetric(horizontal: 10, vertical: 5),
                          decoration: BoxDecoration(color: MyColor.colorWhite, boxShadow: MyUtils.getCardShadow()),
                          child: FaqCardShimmer(),
                        );
                      },
                    ),
                  )
                ],
                if (controller.nomineeList.isNotEmpty) ...[
                  Expanded(
                    child: ListView.separated(
                      separatorBuilder: (context, index) => const SizedBox(height: Dimensions.space15),
                      itemCount: controller.nomineeList.length,
                      itemBuilder: (context, index) => NomineeCard(
                        nominee: controller.nomineeList[index],
                        nomineeImage: "${UrlContainer.nomineeAssetPath.replaceAll('_', controller.nomineeImagePath)}/${controller.nomineeList[index].nomineeImage}",
                        onDeletePress: () {
                          CustomBottomSheet(
                            child: NomineeRemoveBottomSheet(nominee: controller.nomineeList[index]),
                          ).customBottomSheet(context);
                        },
                        onSelectPress: () => controller.selectNominee(controller.nomineeList[index]),
                        isSelected: controller.selectedNominee?.id == controller.nomineeList[index].id,
                      ),
                    ),
                  ),
                ] else if (!controller.isLoading && controller.nomineeFormList.isEmpty) ...[
                  Expanded(
                    child: SingleChildScrollView(
                      child: NoDataWidget(
                        margin: 6,
                        noDataText: MyStrings.noNomineeFound,
                        isAlignmentCenter: true,
                        bottomWidget: Column(
                          children: [
                            Container(
                              margin: EdgeInsets.symmetric(horizontal: 10, vertical: 20),
                              child: GradientRoundedButton(
                                text: MyStrings.addNominee,
                                isLoading: controller.isSubmitLoading,
                                press: () {
                                  Get.toNamed(RouteHelper.addNewNomineeScreen);
                                },
                              ),
                            ),
                          ],
                        ),
                      ),
                    ),
                  )
                ],
              ],
            ),
          );
        },
      ),
      bottomNavigationBar: Container(
        margin: EdgeInsets.symmetric(horizontal: Dimensions.space15, vertical: 5),
        child: GetBuilder<NomineeController>(builder: (nomineeController) {
          return GetBuilder<SavingsController>(
            builder: (controller) {
              return nomineeController.selectedNominee == null || nomineeController.nomineeList.isEmpty
                  ? SizedBox()
                  : GradientRoundedButton(
                      text: MyStrings.continue_,
                      isLoading: controller.isSubmitLoading,
                      press: () {
                        if (nomineeController.selectedNominee == null) {
                          CustomSnackBar.error(errorList: [MyStrings.pleaseSelectANominee]);
                        } else {
                          Get.toNamed(RouteHelper.savingsPinScreen);
                        }
                      },
                    );
            },
          );
        }),
      ),
    );
  }
}
